<?php
/* 
 * RssGen - класс для простой генерации RSS каналов
 * @author Replikon
 * 
 * "Велосипед следует изобретать только в двух случаях:
 * 1. Велосипеда не существует.
 * 2. Все существующие велосипеды с квадратными колесами." (c)
 *
 * Цели, преследовавшиеся во время написания:
 * 1. Легкость создания RSS лент.
 * 2. Легкость использования и Легкость самого класса. Чтобы не был излишне сложным, в отличие от других велосипедов с реактивными двигателями, наведением из космоса и прямой связью с Марсом. Для управления которыми надо пройти 3-месяный курс обучения.
 * 3. Но при этом необходимая функциональность, не как у трехколесного велосипеда.
 * 4. Всегда полностью валидная лента. Никому не хочется, чтобы от велосипеда отваливались куски при езде. И уж точно не хочется потом ремонтировать велосипед.
 * 5. Независимость от фреймворков, CMS и т.п.
 */

class RssGen
{
    /*
     * Название канала
     */
    public $title;

    /*
     * Ссылка на сайт, которому принадлежит эта RSS лента
     * Внимание! Ссылка должна быть абсолютной! Впрочем, как и все ссылки в ленте.
     */
    public $link;

    /*
     * Описание канала
     */
    public $desctiption;

    /*
     * Дата публикации текста в канале в TIMESTAMP.
     * Например, «New York Times» выходит ежедневно и дата публикации меняется каждые 24 часа, тогда и меняется pubDate канала.
     */
    public $pubDate;

    /*
     * Время Последнего изменения содержимого канала в TIMESTAMP.
     */
    public $lastBuildDate;



    // Необязательные элементы, но весьма желательные
    /*
     * Изображение, выводимое рядом с названием ленты в агрегаторе.
     * Точнее ссылка на него.
     *
     * Теория:
     * <image> — дочерний элемент <channel>, содержащий три обязательных и три дополнительных элемента.
     * Обязательные элементы:
     * <url> - URL изображения GIF, JPEG или PNG, представляющего канал.
     * <title> - Описание изображения, которое будет использовано в атрибуте alt элемента HTML img, когда канал будет показан в гипертексте.
     * <link> - URL сайта; изображение канала будет служить ссылкой на этот сайт.
     * (Как правило, <title> и <link> изображения должны совпадать с <title> и <link> канала. (Вот поэтому нужно указать лишь ссылку, а все остальное скрипт допишет сам.)
     */
    public $image;

    /*
     * Язык содержимого ленты.
     * По умолчанию русский.
     */
    public $language = 'ru';






    /*
     * Переменная с содержимым тегов Item.
     * В нее динамически "собирается" канал в процессе работы класса.
     */
    protected $items;



    /*
     * Добавление нового элемента Item
     * @param $item Экземпляр класса RssGenItem
     */
    public function addItem (RssGenItem $item)
    {
        $this->items .= $item->getXml ();
    }

    /*
     * Отображение содержимого ленты
     */
    public function render ()
    {
        header ('Content-type: application/rss+xml');

        echo '<?xml version="1.0" encoding="UTF-8"?>
<rss version="2.0">
<channel>
    <title><![CDATA[' . $this->title . ']]></title>
    <link>' . $this->link . '</link>
    <description><![CDATA[' . $this->desctiption . ']]></description>
    <language>' . $this->language . '</language>
    <generator>' . $this->getGenerator () . '</generator>
    <pubDate>' . date ('r', $this->pubDate) . '</pubDate>
    <lastBuildDate>' . date ('r', $this->lastBuildDate) . '</lastBuildDate>
';

        if (!empty ($this->image)) echo '    <image>
        <url>' . $this->image . '</url>
        <title>' . $this->title . '</title>
        <link>' . $this->link . '</link>
    </image>
';

        echo $this->items;

        echo '
</channel>
</rss>';
    }



    /*
     * Текущая версия класса
     */
    public function getVersion ()
    {
        return '2.0';
    }

    /*
     * Поле Generator в ленте
     */
    public function getGenerator ()
    {
        return 'RssGen v' . $this->getVersion ();
    }
}



/*
 * Класс элемента item
 */
class RssGenItem
{
    /*
     * Заголовок материала
     */
    public $title;

    /*
     * Ссылка на материал
     */
    public $link;

    /*
     * Описание материала
     */
    public $description;



    // Необязательные элементы, но весьма желательные
    /*
     * Дата публикации материала в TIMESTAMP
     */
    public $pubDate;

    /*
     * Строка, уникальным образом идентифицирующая сообщение.
     * По умолчанию, в этот атрибут автоматически вводится ссылка на статью.
     *
     * Немного теории:
     * Guid — уникальный на уровне канала строковый идентификатор статьи. Сборщик может использовать идентификаторы для выявления новых статей.
     *
     * Пример: <guid>http://some.server.com/weblogItem3207</guid>.
     * Синтаксис guid не определен.
     *
     * Если для guid необязательный атрибут isPermaLink имеет значение true, то идентификатор рассматривается как постоянная ссылка на полную статью, описанную в элементе <item>:
     * <guid isPermaLink="true">http://inessential.com/2002/09/01.php#a2</guid>.
     * 
     * ПО УМОЛЧАНИЮ ЗНАЧЕНИЕ isPermaLink РАВНО TRUE; если указано false, то guid не может рассматриваться как URL.
     *
     * Часто возникает вопрос касательно элементов <guid>: это не одно и то же, что и <link>?
     * В некоторых системах, предоставляющих контент, это так, в других — нет.
     * В ряде систем <link> является постоянной ссылкой на запись блога; в других — ссылкой на полный текст (если в <item> содержится краткое изложение), а на запись в блоге ведет <guid>.
     * В любом случае, рекомендуется, чтобы вы указывали guid, и, если возможно, он был постоянной ссылкой. Это позволяет сборщикам не повторять записи, даже если они редактировались.
     */
    public $guid;

    /*
     * URL страницы для комментариев, относящихся к сообщению.
     */
    public $comments;

    /*
     * Элемент указывает адрес электронной почты автора сообщения.
     * Но часто используют просто логин.
     */
    public $author;



    /*
     * Возвращает фрагмент RSS ленты с текущим тегом Item
     */
    public function getXml ()
    {
        if (empty ($this->guid)) $this->guid = $this->link;

        // Собираем элемент
        $xml = '
    <item>
        <title><![CDATA[' . $this->title . ']]></title>
        <link>' . $this->link . '</link>
        <description><![CDATA[' . $this->description . ']]></description>';

        if (!empty ($this->pubDate))    $xml .= '
        <pubDate>' . date ('r', $this->pubDate) . '</pubDate>';
        if (!empty ($this->guid))       $xml .= '
        <guid isPermaLink="true">' . $this->guid . '</guid>';
        if (!empty ($this->comments))   $xml .= '
        <comments>' . $this->comments . '</comments>';
        if (!empty ($this->author))     $xml .= '
        <author>' . $this->author . '</author>';

        $xml .= '
    </item>
';
        return $xml;
    }
}